from django.shortcuts import render, redirect, get_object_or_404
from django.contrib.auth.decorators import login_required
from django.contrib import messages
from django.utils import timezone
from datetime import date
from django.core.mail import send_mail
from django.template.loader import render_to_string
from django.conf import settings
from django.urls import reverse
from app.models import User, StudentProfile
from .models import (
    AnnualSubscriptionDetails, AnnualSubscription,
    SubjectBasedSubscriptionDetails, SubjectBasedSubscription,
    BundleSubscriptionDetails, BundleSubscription
)
from app.models import ClassLevel
# Subscription Pages
def subscriptionPage(request):
    return render(request, 'web/subscription/subscriptions.html')



def AnnualSubscriptionPage(request):
    annual_subscription = AnnualSubscriptionDetails.objects.prefetch_related('keys').all()
    context = {'annual_subscription': annual_subscription}
    return render(request, 'web/subscription/annual_subscription.html', context)


def SubjectSubscriptionPage(request):
    subject_subscription = SubjectBasedSubscriptionDetails.objects.prefetch_related('keys').all()
    class_levels = ClassLevel.objects.all()
    context = {
        'subject_subscription': subject_subscription,
        'class_levels': class_levels,
    }
    return render(request, 'web/subscription/subject_subcription.html', context)



def BundleSubscriptionPage(request):
    bundle_subscription = BundleSubscriptionDetails.objects.prefetch_related('subjects').all()
    context = {'bundle_subscription': bundle_subscription}
    return render(request, 'web/subscription/bundle_subscription.html', context)



# Checkout Pages
def checkoutPageAnnual(request, pk):
    subscription = get_object_or_404(AnnualSubscriptionDetails, pk=pk)
    if subscription.price == 0:
        return redirect('free_subscription', subscription_type='annual', pk=pk)
    
    context = {
        'subscription': subscription,
        'subscription_type': 'annual',
        'current_year': date.today().year,
    }
    return render(request, 'web/subscription/payment.html', context)



@login_required(login_url='login_user')
def checkoutPageSubjectBased(request, pk):
    subscription = get_object_or_404(SubjectBasedSubscriptionDetails, pk=pk)
    if subscription.price == 0:
        return redirect('free_subscription', subscription_type='subject_based', pk=pk)
    context = {
        'subscription': subscription,
        'subscription_type': 'subject_based',
        'current_year': date.today().year,
    }
    return render(request, 'web/subscription/payment.html', context)



@login_required(login_url='login_user')
def checkoutPageBundle(request, pk):
    subscription = get_object_or_404(BundleSubscriptionDetails, pk=pk)
    if subscription.price == 0:
        return redirect('free_subscription', subscription_type='bundle', pk=pk)
    context = {
        'subscription': subscription,
        'subscription_type': 'bundle',
        'current_year': date.today().year,
    }
    return render(request, 'web/subscription/payment.html', context)



@login_required(login_url='login_user')
def process_free_subscription(request, subscription_type, pk):
    if request.method == 'POST':
        try:
            user = request.user
            today = timezone.now().date()
            
            # Get or create student profile
            student_profile = user.studentprofile
            
            if subscription_type == 'annual':
                subscription_details = get_object_or_404(AnnualSubscriptionDetails, pk=pk)
                expiration_date = date(today.year + 1, today.month, today.day)
                subscription = AnnualSubscription.objects.create(
                    annual_subscription_details=subscription_details,
                    student=student_profile,
                    class_level=subscription_details.class_level,
                    start_date=today,
                    end_date=expiration_date,
                    is_active=True
                )
            elif subscription_type == 'subject_based':
                subscription_details = get_object_or_404(SubjectBasedSubscriptionDetails, pk=pk)
                expiration_date = date(today.year, 12, 31)
                subscription = SubjectBasedSubscription.objects.create(
                    subject_based_subscription_details=subscription_details,
                    student=student_profile,
                    class_level=subscription_details.class_level,
                    start_date=today,
                    end_date=expiration_date,
                    is_active=True
                )
                subscription.subjects.add(subscription_details.subject)
            elif subscription_type == 'bundle':
                subscription_details = get_object_or_404(BundleSubscriptionDetails, pk=pk)
                expiration_date = date(today.year, 12, 31)
                subscription = BundleSubscription.objects.create(
                    bundle_subscription_details=subscription_details,
                    student=student_profile,
                    class_level=subscription_details.class_level,
                    start_date=today,
                    end_date=expiration_date,
                    is_active=True
                )
                subscription.subjects.set(subscription_details.subjects.all())
            
            # Send confirmation email
            send_subscription_confirmation(user, subscription, subscription_type)
            
            # Handle Turbo Frame request
            if request.headers.get('Turbo-Frame'):
                return render(request, 'web/subscription/turbo_redirect.html', {
                    'redirect_url': reverse('subscription_page')
                })
            
            messages.success(request, 'বিনামূল্যে সাবস্ক্রিপশনটি আপনার প্রোফাইলে যুক্ত হয়ে গিয়েছে।')
            return redirect('subscription_page')
        
        except Exception as e:
            if request.headers.get('Turbo-Frame'):
                return render(request, 'web/subscription/turbo_error.html', {
                    'message': f'An error occurred: {str(e)}'
                }, status=500)
            
            messages.error(request, f'An error occurred: {str(e)}')
            return redirect('subscription')
    else:
        return redirect('subscription')
    






@login_required(login_url='login_user')
def process_subscription(request):
    if request.method == 'POST':
        try:
            # Get form data
            subscription_type = request.POST.get('subscription_type')
            subscription_id = request.POST.get('subscription_id')
            first_name = request.POST.get('first_name')
            last_name = request.POST.get('last_name')
            email = request.POST.get('email')
            phone = request.POST.get('phone')
            address = request.POST.get('address')
            city = request.POST.get('city')
            payment_method = request.POST.get('payment_method')

            # Validate required fields
            if not all([first_name, last_name, email, phone, address, city, payment_method]):
                messages.error(request, 'Please fill all required fields')
                return redirect(request.META.get('HTTP_REFERER'))

            # Update user profile
            user = request.user
            user.first_name = first_name
            user.last_name = last_name
            user.email = email
            user.phone_number = phone
            user.save()

            # Create/update student profile
            student_profile, created = StudentProfile.objects.get_or_create(user=user)
            student_profile.school_name = f"{address}, {city}"
            student_profile.save()

            today = timezone.now().date()
            subscription = None

            # Handle different subscription types
            if subscription_type == 'annual':
                subscription = create_annual_subscription(subscription_id, student_profile, today)
            elif subscription_type == 'subject_based':
                subscription = create_subject_subscription(subscription_id, student_profile, today)
            elif subscription_type == 'bundle':
                subscription = create_bundle_subscription(subscription_id, student_profile, today)

            if subscription:
                send_subscription_confirmation(user, subscription, subscription_type)
                messages.success(request, 'আপনার সাবস্ক্রিপশন সফলভাবে সম্পন্ন হয়েছে। আপনার প্রোফাইলে সাবস্ক্রিপশনটি যুক্ত হয়ে গিয়েছে। এবং ই-মেইলের মাধ্যমে নিশ্চিতকরণ পাঠানো হয়েছে।')
                return redirect('app:dashboard')

        except Exception as e:
            messages.error(request, f'An error occurred: {str(e)}')
            return redirect(request.META.get('HTTP_REFERER'))

    return redirect('subscription')



def create_annual_subscription(subscription_id, student_profile, today):
    subscription_details = get_object_or_404(AnnualSubscriptionDetails, id=subscription_id)
    expiration_date = date(today.year + 1, today.month, today.day)
    return AnnualSubscription.objects.create(
        annual_subscription_details=subscription_details,
        student=student_profile,
        class_level=subscription_details.class_level,
        start_date=today,
        end_date=expiration_date,
        is_active=True
    )




def create_subject_subscription(subscription_id, student_profile, today):
    subscription_details = get_object_or_404(SubjectBasedSubscriptionDetails, id=subscription_id)
    expiration_date = date(today.year, 12, 31)
    subscription = SubjectBasedSubscription.objects.create(
        subject_based_subscription_details=subscription_details,
        student=student_profile,
        class_level=subscription_details.class_level,
        start_date=today,
        end_date=expiration_date,
        is_active=True
    )
    subscription.subjects.add(subscription_details.subject)
    return subscription




def create_bundle_subscription(subscription_id, student_profile, today):
    subscription_details = get_object_or_404(BundleSubscriptionDetails, id=subscription_id)
    expiration_date = date(today.year, 12, 31)
    subscription = BundleSubscription.objects.create(
        bundle_subscription_details=subscription_details,
        student=student_profile,
        class_level=subscription_details.class_level,
        start_date=today,
        end_date=expiration_date,
        is_active=True
    )
    # Add all subjects from bundle details
    subscription.subjects.set(subscription_details.subjects.all())
    return subscription



# Email Functions
def send_subscription_confirmation(user, subscription, subscription_type):
    """Send confirmation email after purchase"""
    if subscription_type == 'annual':
        details = subscription.annual_subscription_details
    elif subscription_type == 'subject_based':
        details = subscription.subject_based_subscription_details
    else:  # bundle
        details = subscription.bundle_subscription_details
    
    context = {
        'user': user,
        'subscription': subscription,
        'subscription_type': subscription_type,
        'subscription_title': details.title,
        'start_date': subscription.start_date.strftime("%B %d, %Y"),
        'end_date': subscription.end_date.strftime("%B %d, %Y"),
        'duration': '1 Year' if subscription_type == 'annual' else 'Until Year End',
        'subjects': [s.name for s in subscription.subjects.all()] if hasattr(subscription, 'subjects') else [],
        'dashboard_url': reverse('dashboard')
    }
    
    subject = f"Your {subscription_type} Subscription Confirmation"
    html_message = render_to_string('emails/subscription_confirmation.html', context)
    
    send_mail(
        subject,
        "Please view this email in HTML format",
        settings.DEFAULT_FROM_EMAIL,
        [user.email],
        html_message=html_message,
        fail_silently=False
    )

def send_expiry_notification(subscription, days_left=7):
    """Send notification before subscription expires"""
    user = subscription.student.user
    subscription_type = subscription.__class__.__name__.replace('Subscription', '').lower()
    
    context = {
        'user': user,
        'subscription': subscription,
        'subscription_type': subscription_type,
        'days_left': days_left,
        'expiry_date': subscription.end_date.strftime("%B %d, %Y"),
        'renew_url': reverse('renew_subscription', kwargs={'pk': subscription.pk})
    }
    
    subject = f"Your subscription expires in {days_left} days"
    html_message = render_to_string('emails/subscription_expiring_soon.html', context)
    
    send_mail(
        subject,
        "Please view this email in HTML format",
        settings.DEFAULT_FROM_EMAIL,
        [user.email],
        html_message=html_message,
        fail_silently=False
    )