"""
Populate database with real tech-related data
Topics: Web Development, Python, Django, Networking, Template Design
"""
import os
import django

os.environ.setdefault('DJANGO_SETTINGS_MODULE', 'passMCQ.settings')
django.setup()

from app.models import ClassLevel, Subject, Chapter, Question, Choice, ExamCategory, User, StudentProfile
from subscription.models import BundleSubscriptionDetails
from django.utils import timezone
from datetime import timedelta

def clear_data():
    """Clear all existing data"""
    print("Clearing existing data...")
    Choice.objects.all().delete()
    Question.objects.all().delete()
    Chapter.objects.all().delete()
    Subject.objects.all().delete()
    ClassLevel.objects.all().delete()
    ExamCategory.objects.all().delete()
    BundleSubscriptionDetails.objects.all().delete()
    print("✓ Data cleared")

def create_class_levels():
    """Create class levels for tech courses"""
    print("\nCreating class levels...")
    classes = [
        {"name": "Beginner", "order": 1},
        {"name": "Intermediate", "order": 2},
        {"name": "Advanced", "order": 3},
        {"name": "Expert", "order": 4},
    ]
    
    class_objs = {}
    for cls in classes:
        obj = ClassLevel.objects.create(**cls)
        class_objs[cls["name"]] = obj
        print(f"  ✓ Created {cls['name']}")
    
    return class_objs

def create_exam_categories():
    """Create exam categories"""
    print("\nCreating exam categories...")
    categories = [
        "Quick Quiz",
        "Practice Test",
        "Mock Exam",
        "Assessment",
        "Challenge"
    ]
    
    cat_objs = {}
    for cat_name in categories:
        obj = ExamCategory.objects.create(name=cat_name)
        cat_objs[cat_name] = obj
        print(f"  ✓ Created {cat_name}")
    
    return cat_objs

def create_subjects_and_chapters(classes):
    """Create tech subjects and their chapters"""
    print("\nCreating subjects and chapters...")
    
    subjects_data = {
        "Beginner": [
            {
                "name": "HTML & CSS Basics",
                "chapters": [
                    "HTML Fundamentals",
                    "CSS Styling",
                    "Responsive Design",
                    "Flexbox & Grid"
                ]
            },
            {
                "name": "Python Fundamentals",
                "chapters": [
                    "Variables and Data Types",
                    "Control Flow",
                    "Functions",
                    "Lists and Tuples"
                ]
            },
            {
                "name": "Basic Networking",
                "chapters": [
                    "Network Basics",
                    "IP Addressing",
                    "HTTP/HTTPS",
                    "DNS Fundamentals"
                ]
            }
        ],
        "Intermediate": [
            {
                "name": "JavaScript & DOM",
                "chapters": [
                    "JavaScript Basics",
                    "DOM Manipulation",
                    "Events & Event Handlers",
                    "Async JavaScript"
                ]
            },
            {
                "name": "Python Advanced",
                "chapters": [
                    "Object-Oriented Programming",
                    "File Handling",
                    "Exception Handling",
                    "Modules and Packages"
                ]
            },
            {
                "name": "Web Development",
                "chapters": [
                    "Frontend Frameworks",
                    "Backend Basics",
                    "RESTful APIs",
                    "Database Integration"
                ]
            }
        ],
        "Advanced": [
            {
                "name": "Django Framework",
                "chapters": [
                    "Django Architecture",
                    "Models & ORM",
                    "Views & Templates",
                    "Forms & Authentication",
                    "Django REST Framework",
                    "Deployment"
                ]
            },
            {
                "name": "Advanced Networking",
                "chapters": [
                    "TCP/IP Protocol",
                    "Network Security",
                    "Load Balancing",
                    "CDN & Caching"
                ]
            },
            {
                "name": "Modern Web Design",
                "chapters": [
                    "UI/UX Principles",
                    "Design Systems",
                    "CSS Frameworks",
                    "Animation & Transitions"
                ]
            }
        ],
        "Expert": [
            {
                "name": "Full Stack Development",
                "chapters": [
                    "Microservices Architecture",
                    "Docker & Kubernetes",
                    "CI/CD Pipelines",
                    "Cloud Deployment"
                ]
            },
            {
                "name": "Advanced Django",
                "chapters": [
                    "Custom Middleware",
                    "Signals & Celery",
                    "Performance Optimization",
                    "Security Best Practices"
                ]
            }
        ]
    }
    
    all_subjects = {}
    for class_name, subjects in subjects_data.items():
        class_obj = classes[class_name]
        for subject_data in subjects:
            subject = Subject.objects.create(
                name=subject_data["name"],
                class_level=class_obj
            )
            all_subjects[subject_data["name"]] = {
                "subject": subject,
                "chapters": []
            }
            print(f"  ✓ Created subject: {subject_data['name']}")
            
            for chapter_name in subject_data["chapters"]:
                chapter = Chapter.objects.create(
                    title=chapter_name,
                    subject=subject,
                    class_level=class_obj
                )
                all_subjects[subject_data["name"]]["chapters"].append(chapter)
                print(f"    → Chapter: {chapter_name}")
    
    return all_subjects

def create_questions(subjects, categories):
    """Create real tech questions with correct answers"""
    print("\nCreating questions...")
    
    # Web Development Questions
    web_questions = [
        {
            "subject": "HTML & CSS Basics",
            "chapter": "HTML Fundamentals",
            "text": "What does HTML stand for?",
            "difficulty": 1,
            "choices": [
                {"text": "Hyper Text Markup Language", "is_correct": True, "explanation": "HTML stands for Hyper Text Markup Language, which is the standard markup language for creating web pages."},
                {"text": "High Tech Modern Language", "is_correct": False, "explanation": "This is incorrect."},
                {"text": "Home Tool Markup Language", "is_correct": False, "explanation": "This is incorrect."},
                {"text": "Hyperlinks and Text Markup Language", "is_correct": False, "explanation": "This is incorrect."}
            ]
        },
        {
            "subject": "HTML & CSS Basics",
            "chapter": "CSS Styling",
            "text": "Which CSS property is used to change text color?",
            "difficulty": 1,
            "choices": [
                {"text": "color", "is_correct": True, "explanation": "The 'color' property in CSS is used to set the color of text."},
                {"text": "text-color", "is_correct": False, "explanation": "This is not a valid CSS property."},
                {"text": "font-color", "is_correct": False, "explanation": "This is not a valid CSS property."},
                {"text": "text-style", "is_correct": False, "explanation": "This property doesn't exist in CSS."}
            ]
        },
        {
            "subject": "HTML & CSS Basics",
            "chapter": "Responsive Design",
            "text": "Which meta tag is essential for responsive web design?",
            "difficulty": 2,
            "choices": [
                {"text": "<meta name='viewport' content='width=device-width, initial-scale=1'>", "is_correct": True, "explanation": "The viewport meta tag is essential for responsive design as it controls the page's dimensions and scaling on mobile devices."},
                {"text": "<meta charset='UTF-8'>", "is_correct": False, "explanation": "This sets character encoding, not viewport settings."},
                {"text": "<meta name='description'>", "is_correct": False, "explanation": "This is for SEO, not responsive design."},
                {"text": "<meta http-equiv='refresh'>", "is_correct": False, "explanation": "This is for page refresh, not responsive design."}
            ]
        },
        {
            "subject": "HTML & CSS Basics",
            "chapter": "Flexbox & Grid",
            "text": "Which CSS property creates a flex container?",
            "difficulty": 2,
            "choices": [
                {"text": "display: flex;", "is_correct": True, "explanation": "Setting display: flex on an element makes it a flex container and its children become flex items."},
                {"text": "flex: container;", "is_correct": False, "explanation": "This is not valid CSS syntax."},
                {"text": "position: flex;", "is_correct": False, "explanation": "flex is not a valid position value."},
                {"text": "container: flex;", "is_correct": False, "explanation": "This property doesn't exist."}
            ]
        },
        {
            "subject": "JavaScript & DOM",
            "chapter": "JavaScript Basics",
            "text": "Which keyword is used to declare a block-scoped variable in JavaScript?",
            "difficulty": 2,
            "choices": [
                {"text": "let", "is_correct": True, "explanation": "'let' declares a block-scoped variable, meaning it's only available within the block it's defined in."},
                {"text": "var", "is_correct": False, "explanation": "'var' is function-scoped, not block-scoped."},
                {"text": "const", "is_correct": False, "explanation": "While 'const' is block-scoped, the question asks for variables that can be reassigned."},
                {"text": "define", "is_correct": False, "explanation": "This is not a JavaScript keyword for variable declaration."}
            ]
        },
        {
            "subject": "JavaScript & DOM",
            "chapter": "DOM Manipulation",
            "text": "Which method is used to select an element by its ID in JavaScript?",
            "difficulty": 1,
            "choices": [
                {"text": "document.getElementById()", "is_correct": True, "explanation": "document.getElementById() is the standard method to select an element by its ID attribute."},
                {"text": "document.querySelector()", "is_correct": False, "explanation": "While this can select by ID using '#id', getElementById() is the specific method for IDs."},
                {"text": "document.getElement()", "is_correct": False, "explanation": "This method doesn't exist in JavaScript."},
                {"text": "document.selectById()", "is_correct": False, "explanation": "This is not a valid JavaScript method."}
            ]
        },
        {
            "subject": "Web Development",
            "chapter": "RESTful APIs",
            "text": "Which HTTP method is used to retrieve data from a server?",
            "difficulty": 2,
            "choices": [
                {"text": "GET", "is_correct": True, "explanation": "GET is the HTTP method used to retrieve data from a server without modifying it."},
                {"text": "POST", "is_correct": False, "explanation": "POST is used to send data to create a new resource."},
                {"text": "PUT", "is_correct": False, "explanation": "PUT is used to update an existing resource."},
                {"text": "DELETE", "is_correct": False, "explanation": "DELETE is used to remove a resource."}
            ]
        }
    ]
    
    # Python Questions
    python_questions = [
        {
            "subject": "Python Fundamentals",
            "chapter": "Variables and Data Types",
            "text": "What is the output of: type(5.0)",
            "difficulty": 1,
            "choices": [
                {"text": "<class 'float'>", "is_correct": True, "explanation": "5.0 is a floating-point number, so type() returns <class 'float'>."},
                {"text": "<class 'int'>", "is_correct": False, "explanation": "5.0 has a decimal point, making it a float, not an integer."},
                {"text": "<class 'number'>", "is_correct": False, "explanation": "There is no 'number' type in Python."},
                {"text": "<class 'decimal'>", "is_correct": False, "explanation": "The base type is 'float', not 'decimal'."}
            ]
        },
        {
            "subject": "Python Fundamentals",
            "chapter": "Control Flow",
            "text": "Which statement is used to exit a loop prematurely?",
            "difficulty": 1,
            "choices": [
                {"text": "break", "is_correct": True, "explanation": "'break' exits the loop immediately, regardless of the loop condition."},
                {"text": "continue", "is_correct": False, "explanation": "'continue' skips to the next iteration but doesn't exit the loop."},
                {"text": "exit", "is_correct": False, "explanation": "'exit' terminates the program, not just the loop."},
                {"text": "stop", "is_correct": False, "explanation": "Python doesn't have a 'stop' keyword."}
            ]
        },
        {
            "subject": "Python Fundamentals",
            "chapter": "Functions",
            "text": "What keyword is used to define a function in Python?",
            "difficulty": 1,
            "choices": [
                {"text": "def", "is_correct": True, "explanation": "'def' is the keyword used to define functions in Python."},
                {"text": "function", "is_correct": False, "explanation": "Python uses 'def', not 'function'."},
                {"text": "func", "is_correct": False, "explanation": "'func' is not a Python keyword."},
                {"text": "define", "is_correct": False, "explanation": "Python doesn't use 'define' for functions."}
            ]
        },
        {
            "subject": "Python Fundamentals",
            "chapter": "Lists and Tuples",
            "text": "What is the main difference between lists and tuples?",
            "difficulty": 2,
            "choices": [
                {"text": "Lists are mutable, tuples are immutable", "is_correct": True, "explanation": "Lists can be modified after creation (mutable), while tuples cannot be changed once created (immutable)."},
                {"text": "Lists are faster than tuples", "is_correct": False, "explanation": "Tuples are actually faster than lists due to their immutability."},
                {"text": "Tuples can only store numbers", "is_correct": False, "explanation": "Tuples can store any type of data, just like lists."},
                {"text": "Lists use () and tuples use []", "is_correct": False, "explanation": "It's the opposite: lists use [] and tuples use ()."}
            ]
        },
        {
            "subject": "Python Advanced",
            "chapter": "Object-Oriented Programming",
            "text": "What is a constructor method called in Python?",
            "difficulty": 2,
            "choices": [
                {"text": "__init__", "is_correct": True, "explanation": "__init__ is the constructor method that initializes new objects in Python classes."},
                {"text": "__new__", "is_correct": False, "explanation": "While __new__ creates instances, __init__ is the initializer/constructor."},
                {"text": "constructor", "is_correct": False, "explanation": "Python doesn't use 'constructor' as a method name."},
                {"text": "__create__", "is_correct": False, "explanation": "This is not a valid Python special method."}
            ]
        },
        {
            "subject": "Python Advanced",
            "chapter": "Exception Handling",
            "text": "Which block is always executed in exception handling?",
            "difficulty": 2,
            "choices": [
                {"text": "finally", "is_correct": True, "explanation": "The 'finally' block always executes, whether an exception occurred or not."},
                {"text": "except", "is_correct": False, "explanation": "'except' only executes when an exception is caught."},
                {"text": "try", "is_correct": False, "explanation": "'try' contains code that might raise an exception."},
                {"text": "catch", "is_correct": False, "explanation": "Python uses 'except', not 'catch'."}
            ]
        }
    ]
    
    # Django Questions
    django_questions = [
        {
            "subject": "Django Framework",
            "chapter": "Django Architecture",
            "text": "What architectural pattern does Django follow?",
            "difficulty": 2,
            "choices": [
                {"text": "MVT (Model-View-Template)", "is_correct": True, "explanation": "Django follows the MVT pattern: Model for data, View for logic, Template for presentation."},
                {"text": "MVC (Model-View-Controller)", "is_correct": False, "explanation": "While similar, Django officially uses MVT, not MVC."},
                {"text": "MVVM (Model-View-ViewModel)", "is_correct": False, "explanation": "MVVM is used in frameworks like Angular, not Django."},
                {"text": "MVP (Model-View-Presenter)", "is_correct": False, "explanation": "MVP is different from Django's MVT pattern."}
            ]
        },
        {
            "subject": "Django Framework",
            "chapter": "Models & ORM",
            "text": "Which method is used to create a new record in Django ORM?",
            "difficulty": 2,
            "choices": [
                {"text": "Model.objects.create()", "is_correct": True, "explanation": "create() both instantiates and saves a new object to the database in one step."},
                {"text": "Model.add()", "is_correct": False, "explanation": "Django ORM doesn't have an 'add' method."},
                {"text": "Model.insert()", "is_correct": False, "explanation": "Django uses 'create()', not 'insert()'."},
                {"text": "Model.new()", "is_correct": False, "explanation": "This is not a valid Django ORM method."}
            ]
        },
        {
            "subject": "Django Framework",
            "chapter": "Views & Templates",
            "text": "What is the purpose of {% csrf_token %} in Django templates?",
            "difficulty": 3,
            "choices": [
                {"text": "Protects against Cross-Site Request Forgery attacks", "is_correct": True, "explanation": "{% csrf_token %} adds a hidden token to forms to prevent CSRF attacks by verifying the request origin."},
                {"text": "Encrypts form data", "is_correct": False, "explanation": "It doesn't encrypt data, it validates request authenticity."},
                {"text": "Validates user input", "is_correct": False, "explanation": "Input validation is separate from CSRF protection."},
                {"text": "Creates user sessions", "is_correct": False, "explanation": "Session creation is handled separately by Django's session framework."}
            ]
        },
        {
            "subject": "Django Framework",
            "chapter": "Forms & Authentication",
            "text": "Which decorator restricts view access to logged-in users?",
            "difficulty": 2,
            "choices": [
                {"text": "@login_required", "is_correct": True, "explanation": "@login_required decorator ensures only authenticated users can access the view."},
                {"text": "@authenticate", "is_correct": False, "explanation": "This decorator doesn't exist in Django."},
                {"text": "@user_required", "is_correct": False, "explanation": "The correct decorator is @login_required."},
                {"text": "@permission_required", "is_correct": False, "explanation": "This checks for specific permissions, not just login status."}
            ]
        },
        {
            "subject": "Django Framework",
            "chapter": "Django REST Framework",
            "text": "What is the purpose of serializers in Django REST Framework?",
            "difficulty": 3,
            "choices": [
                {"text": "Convert complex data types to JSON and vice versa", "is_correct": True, "explanation": "Serializers transform Django models and querysets into JSON/XML formats and validate incoming data."},
                {"text": "Handle URL routing", "is_correct": False, "explanation": "URL routing is handled by URLconf, not serializers."},
                {"text": "Manage database connections", "is_correct": False, "explanation": "Database connections are managed by Django's ORM."},
                {"text": "Create API documentation", "is_correct": False, "explanation": "Documentation is generated by tools like drf-spectacular, not serializers."}
            ]
        },
        {
            "subject": "Advanced Django",
            "chapter": "Custom Middleware",
            "text": "What is the correct order of middleware execution in Django?",
            "difficulty": 4,
            "choices": [
                {"text": "Request: top to bottom, Response: bottom to top", "is_correct": True, "explanation": "Middleware is executed top-to-bottom for requests and bottom-to-top for responses, forming an onion-like structure."},
                {"text": "Always top to bottom", "is_correct": False, "explanation": "Response processing goes in reverse order."},
                {"text": "Random order based on priority", "is_correct": False, "explanation": "Middleware follows a strict order defined in settings."},
                {"text": "Always bottom to top", "is_correct": False, "explanation": "Request processing goes top to bottom."}
            ]
        },
        {
            "subject": "Advanced Django",
            "chapter": "Signals & Celery",
            "text": "What is the purpose of Django signals?",
            "difficulty": 3,
            "choices": [
                {"text": "Allow decoupled applications to get notified of actions", "is_correct": True, "explanation": "Signals allow certain senders to notify receivers when actions occur, enabling loose coupling between components."},
                {"text": "Handle asynchronous tasks", "is_correct": False, "explanation": "Celery handles async tasks; signals are for synchronous notifications."},
                {"text": "Manage database transactions", "is_correct": False, "explanation": "Transaction management is separate from signals."},
                {"text": "Route URL requests", "is_correct": False, "explanation": "URL routing is handled by URLconf."}
            ]
        },
        {
            "subject": "Advanced Django",
            "chapter": "Performance Optimization",
            "text": "Which method prevents the N+1 query problem in Django?",
            "difficulty": 4,
            "choices": [
                {"text": "select_related() and prefetch_related()", "is_correct": True, "explanation": "These methods optimize queries by reducing the number of database hits through JOIN operations and prefetching."},
                {"text": "filter() and exclude()", "is_correct": False, "explanation": "These filter querysets but don't optimize related object queries."},
                {"text": "all() and get()", "is_correct": False, "explanation": "These basic query methods don't optimize related object access."},
                {"text": "count() and exists()", "is_correct": False, "explanation": "These check for object existence but don't optimize related queries."}
            ]
        }
    ]
    
    # Networking Questions
    networking_questions = [
        {
            "subject": "Basic Networking",
            "chapter": "Network Basics",
            "text": "What does IP stand for in networking?",
            "difficulty": 1,
            "choices": [
                {"text": "Internet Protocol", "is_correct": True, "explanation": "IP stands for Internet Protocol, which is the principal protocol for routing data across networks."},
                {"text": "Internal Process", "is_correct": False, "explanation": "This is incorrect."},
                {"text": "Information Package", "is_correct": False, "explanation": "This is incorrect."},
                {"text": "Integrated Protocol", "is_correct": False, "explanation": "This is incorrect."}
            ]
        },
        {
            "subject": "Basic Networking",
            "chapter": "IP Addressing",
            "text": "What is the default subnet mask for a Class C network?",
            "difficulty": 3,
            "choices": [
                {"text": "255.255.255.0", "is_correct": True, "explanation": "Class C networks use 255.255.255.0 as the default subnet mask, providing 254 usable host addresses."},
                {"text": "255.255.0.0", "is_correct": False, "explanation": "This is the default mask for Class B networks."},
                {"text": "255.0.0.0", "is_correct": False, "explanation": "This is the default mask for Class A networks."},
                {"text": "255.255.255.255", "is_correct": False, "explanation": "This is a broadcast address mask, not a network mask."}
            ]
        },
        {
            "subject": "Basic Networking",
            "chapter": "HTTP/HTTPS",
            "text": "What port does HTTPS typically use?",
            "difficulty": 2,
            "choices": [
                {"text": "443", "is_correct": True, "explanation": "HTTPS uses port 443 by default for secure web traffic."},
                {"text": "80", "is_correct": False, "explanation": "Port 80 is used for HTTP, not HTTPS."},
                {"text": "8080", "is_correct": False, "explanation": "8080 is an alternative HTTP port, not the standard HTTPS port."},
                {"text": "22", "is_correct": False, "explanation": "Port 22 is used for SSH, not HTTPS."}
            ]
        },
        {
            "subject": "Basic Networking",
            "chapter": "DNS Fundamentals",
            "text": "What does DNS stand for?",
            "difficulty": 1,
            "choices": [
                {"text": "Domain Name System", "is_correct": True, "explanation": "DNS (Domain Name System) translates human-readable domain names into IP addresses."},
                {"text": "Digital Network Service", "is_correct": False, "explanation": "This is incorrect."},
                {"text": "Data Name Server", "is_correct": False, "explanation": "This is incorrect."},
                {"text": "Domain Network Security", "is_correct": False, "explanation": "This is incorrect."}
            ]
        },
        {
            "subject": "Advanced Networking",
            "chapter": "TCP/IP Protocol",
            "text": "Which layer of the OSI model does TCP operate at?",
            "difficulty": 3,
            "choices": [
                {"text": "Transport Layer", "is_correct": True, "explanation": "TCP operates at Layer 4 (Transport Layer) of the OSI model, providing reliable, ordered delivery of data."},
                {"text": "Network Layer", "is_correct": False, "explanation": "Network Layer (Layer 3) is for IP, not TCP."},
                {"text": "Application Layer", "is_correct": False, "explanation": "Application Layer (Layer 7) is for protocols like HTTP, not TCP."},
                {"text": "Data Link Layer", "is_correct": False, "explanation": "Data Link Layer (Layer 2) handles node-to-node transfer, not TCP."}
            ]
        },
        {
            "subject": "Advanced Networking",
            "chapter": "Network Security",
            "text": "What is the purpose of a firewall?",
            "difficulty": 2,
            "choices": [
                {"text": "Filter network traffic based on security rules", "is_correct": True, "explanation": "Firewalls monitor and control incoming/outgoing network traffic based on predetermined security rules."},
                {"text": "Speed up internet connection", "is_correct": False, "explanation": "Firewalls provide security, not speed optimization."},
                {"text": "Store web pages", "is_correct": False, "explanation": "Web page storage is done by web servers, not firewalls."},
                {"text": "Translate domain names", "is_correct": False, "explanation": "Domain name translation is done by DNS, not firewalls."}
            ]
        },
        {
            "subject": "Advanced Networking",
            "chapter": "Load Balancing",
            "text": "What is the primary purpose of load balancing?",
            "difficulty": 3,
            "choices": [
                {"text": "Distribute network traffic across multiple servers", "is_correct": True, "explanation": "Load balancing distributes incoming traffic across multiple servers to ensure no single server is overwhelmed."},
                {"text": "Encrypt network data", "is_correct": False, "explanation": "Encryption is handled by SSL/TLS, not load balancing."},
                {"text": "Monitor server health", "is_correct": False, "explanation": "While load balancers may monitor health, their primary purpose is traffic distribution."},
                {"text": "Backup server data", "is_correct": False, "explanation": "Data backup is separate from load balancing."}
            ]
        }
    ]
    
    # Template Design Questions
    design_questions = [
        {
            "subject": "Modern Web Design",
            "chapter": "UI/UX Principles",
            "text": "What does UX stand for?",
            "difficulty": 1,
            "choices": [
                {"text": "User Experience", "is_correct": True, "explanation": "UX (User Experience) refers to how users interact with and experience a product or service."},
                {"text": "User Extension", "is_correct": False, "explanation": "This is incorrect."},
                {"text": "Universal Exchange", "is_correct": False, "explanation": "This is incorrect."},
                {"text": "Unified Experience", "is_correct": False, "explanation": "This is incorrect."}
            ]
        },
        {
            "subject": "Modern Web Design",
            "chapter": "Design Systems",
            "text": "What is a design system?",
            "difficulty": 2,
            "choices": [
                {"text": "A collection of reusable components and guidelines", "is_correct": True, "explanation": "A design system is a comprehensive set of standards, components, and tools to ensure consistency across products."},
                {"text": "A type of CSS framework", "is_correct": False, "explanation": "While CSS frameworks may be part of a design system, they're not the same thing."},
                {"text": "A graphic design software", "is_correct": False, "explanation": "Design systems are conceptual frameworks, not software."},
                {"text": "A color palette", "is_correct": False, "explanation": "Color palettes are just one component of a design system."}
            ]
        },
        {
            "subject": "Modern Web Design",
            "chapter": "CSS Frameworks",
            "text": "Which of these is a popular CSS framework?",
            "difficulty": 1,
            "choices": [
                {"text": "Bootstrap", "is_correct": True, "explanation": "Bootstrap is one of the most popular CSS frameworks for responsive web development."},
                {"text": "JavaScript", "is_correct": False, "explanation": "JavaScript is a programming language, not a CSS framework."},
                {"text": "MySQL", "is_correct": False, "explanation": "MySQL is a database system, not a CSS framework."},
                {"text": "Apache", "is_correct": False, "explanation": "Apache is a web server, not a CSS framework."}
            ]
        },
        {
            "subject": "Modern Web Design",
            "chapter": "Animation & Transitions",
            "text": "Which CSS property is used for transitions?",
            "difficulty": 2,
            "choices": [
                {"text": "transition", "is_correct": True, "explanation": "The 'transition' property enables smooth changes between CSS property values over a specified duration."},
                {"text": "animation", "is_correct": False, "explanation": "While 'animation' creates animations, 'transition' is for property changes."},
                {"text": "transform", "is_correct": False, "explanation": "'transform' changes element appearance but doesn't control timing."},
                {"text": "motion", "is_correct": False, "explanation": "There is no 'motion' property in CSS."}
            ]
        }
    ]
    
    # Full Stack Questions
    fullstack_questions = [
        {
            "subject": "Full Stack Development",
            "chapter": "Microservices Architecture",
            "text": "What is a key characteristic of microservices?",
            "difficulty": 4,
            "choices": [
                {"text": "Independently deployable services", "is_correct": True, "explanation": "Microservices are small, autonomous services that can be deployed independently, allowing for scalability and flexibility."},
                {"text": "Single large application", "is_correct": False, "explanation": "This describes monolithic architecture, not microservices."},
                {"text": "Shared database for all services", "is_correct": False, "explanation": "Microservices typically have separate databases per service."},
                {"text": "Tightly coupled components", "is_correct": False, "explanation": "Microservices are loosely coupled, not tightly coupled."}
            ]
        },
        {
            "subject": "Full Stack Development",
            "chapter": "Docker & Kubernetes",
            "text": "What is Docker primarily used for?",
            "difficulty": 3,
            "choices": [
                {"text": "Containerization of applications", "is_correct": True, "explanation": "Docker packages applications and their dependencies into containers for consistent deployment across environments."},
                {"text": "Database management", "is_correct": False, "explanation": "Docker is for containerization, not specifically database management."},
                {"text": "Version control", "is_correct": False, "explanation": "Version control is handled by tools like Git, not Docker."},
                {"text": "Code compilation", "is_correct": False, "explanation": "Docker runs applications but doesn't compile code."}
            ]
        },
        {
            "subject": "Full Stack Development",
            "chapter": "CI/CD Pipelines",
            "text": "What does CI/CD stand for?",
            "difficulty": 2,
            "choices": [
                {"text": "Continuous Integration/Continuous Deployment", "is_correct": True, "explanation": "CI/CD automates the integration of code changes and their deployment to production environments."},
                {"text": "Code Integration/Code Deployment", "is_correct": False, "explanation": "The 'C' stands for Continuous, not Code."},
                {"text": "Continuous Installation/Continuous Distribution", "is_correct": False, "explanation": "It's Integration and Deployment, not Installation and Distribution."},
                {"text": "Container Integration/Container Delivery", "is_correct": False, "explanation": "CI/CD is about continuous processes, not specifically containers."}
            ]
        },
        {
            "subject": "Full Stack Development",
            "chapter": "Cloud Deployment",
            "text": "Which is NOT a major cloud service provider?",
            "difficulty": 2,
            "choices": [
                {"text": "WordPress", "is_correct": True, "explanation": "WordPress is a CMS platform, not a cloud service provider like AWS, Azure, or GCP."},
                {"text": "AWS", "is_correct": False, "explanation": "AWS (Amazon Web Services) is a major cloud provider."},
                {"text": "Microsoft Azure", "is_correct": False, "explanation": "Azure is Microsoft's cloud computing platform."},
                {"text": "Google Cloud Platform", "is_correct": False, "explanation": "GCP is Google's cloud service offering."}
            ]
        }
    ]
    
    # Combine all questions
    all_questions = (
        web_questions + 
        python_questions + 
        django_questions + 
        networking_questions + 
        design_questions +
        fullstack_questions
    )
    
    question_count = 0
    for q_data in all_questions:
        subject_info = subjects.get(q_data["subject"])
        if not subject_info:
            continue
            
        subject_obj = subject_info["subject"]
        chapter_obj = next((ch for ch in subject_info["chapters"] if ch.title == q_data["chapter"]), None)
        
        if not chapter_obj:
            continue
        
        # Get random category
        category = categories[list(categories.keys())[question_count % len(categories)]]
        
        question = Question.objects.create(
            text=q_data["text"],
            question_type="MCQ",
            subject=subject_obj,
            chapter=chapter_obj,
            exam_category=category,
            difficulty=q_data["difficulty"]
        )
        
        for choice_data in q_data["choices"]:
            Choice.objects.create(
                question=question,
                text=choice_data["text"],
                is_correct=choice_data["is_correct"],
                explanation=choice_data["explanation"]
            )
        
        question_count += 1
        if question_count % 10 == 0:
            print(f"  ✓ Created {question_count} questions...")
    
    print(f"\n✓ Total questions created: {question_count}")

def create_bundle_subscriptions(classes):
    """Create subscription bundles"""
    print("\nCreating subscription bundles...")
    
    bundles = [
        {
            "title": "Web Development Mastery",
            "description": "Complete web development bundle from basics to advanced. Includes HTML, CSS, JavaScript, and responsive design.",
            "class_level": classes["Beginner"],
            "price": 999
        },
        {
            "title": "Python Pro Package",
            "description": "Master Python programming from fundamentals to advanced concepts. Perfect for aspiring developers.",
            "class_level": classes["Intermediate"],
            "price": 1499
        },
        {
            "title": "Django Developer Bundle",
            "description": "Become a Django expert with comprehensive training. Build scalable web applications with confidence.",
            "class_level": classes["Advanced"],
            "price": 1999
        },
        {
            "title": "Full Stack Elite",
            "description": "Complete full stack development with deployment skills. Master microservices, Docker, and cloud platforms.",
            "class_level": classes["Expert"],
            "price": 2999
        }
    ]
    
    for bundle_data in bundles:
        BundleSubscriptionDetails.objects.create(**bundle_data)
        print(f"  ✓ Created bundle: {bundle_data['title']}")

def create_admin_user():
    """Create admin user"""
    print("\nCreating admin user...")
    if not User.objects.filter(username='admin').exists():
        user = User.objects.create_superuser(
            username='admin',
            email='admin@passmcq.com',
            password='admin123',
            first_name='Admin',
            last_name='User'
        )
        StudentProfile.objects.create(
            user=user,
            school_name='PassMCQ Admin',
            current_class=ClassLevel.objects.first()
        )
        print("  ✓ Admin user created (username: admin, password: admin123)")
    else:
        print("  ℹ Admin user already exists")

def create_test_users(classes):
    """Create test users"""
    print("\nCreating test users...")
    test_users = [
        {
            "username": "student1",
            "email": "student1@test.com",
            "password": "test123",
            "first_name": "John",
            "last_name": "Doe",
            "class_level": classes["Beginner"]
        },
        {
            "username": "student2",
            "email": "student2@test.com",
            "password": "test123",
            "first_name": "Jane",
            "last_name": "Smith",
            "class_level": classes["Intermediate"]
        }
    ]
    
    for user_data in test_users:
        if not User.objects.filter(username=user_data["username"]).exists():
            class_level = user_data.pop("class_level")
            user = User.objects.create_user(**user_data)
            StudentProfile.objects.create(
                user=user,
                school_name='Tech University',
                current_class=class_level
            )
            print(f"  ✓ Created user: {user_data['username']}")
        else:
            print(f"  ℹ User {user_data['username']} already exists")

def main():
    """Main execution function"""
    print("=" * 60)
    print("PASSMCQ DATABASE POPULATION - TECH CONTENT")
    print("=" * 60)
    
    clear_data()
    classes = create_class_levels()
    categories = create_exam_categories()
    subjects = create_subjects_and_chapters(classes)
    create_questions(subjects, categories)
    create_bundle_subscriptions(classes)
    create_admin_user()
    create_test_users(classes)
    
    print("\n" + "=" * 60)
    print("✓ DATABASE POPULATION COMPLETE!")
    print("=" * 60)
    print("\nSummary:")
    print(f"  • Class Levels: {ClassLevel.objects.count()}")
    print(f"  • Subjects: {Subject.objects.count()}")
    print(f"  • Chapters: {Chapter.objects.count()}")
    print(f"  • Questions: {Question.objects.count()}")
    print(f"  • Choices: {Choice.objects.count()}")
    print(f"  • Exam Categories: {ExamCategory.objects.count()}")
    print(f"  • Users: {User.objects.count()}")
    print("\nAdmin Login:")
    print("  Username: admin")
    print("  Password: admin123")
    print("\nTest User Login:")
    print("  Username: student1 / student2")
    print("  Password: test123")

if __name__ == '__main__':
    main()
